<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Course;
use App\Models\Session;
use App\Models\SessionFile as File;
use App\Http\Resources\CourseResource;
use App\Traits\GenericApiResponse;

class CoursesController extends Controller
{
    use GenericApiResponse;

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        if($request->user()->role === 'peserta'){
            $courses = Course::whereHas('users', function($q) use($request){
                $q->where('id', $request->user()->id);
            })
            ->orderBy('created_at', 'ASC')
            ->paginate($request->per_page ?? 10);
        }else{
            $is_online = $request->input('is_online') ? $request->input('is_online'):'a';

            if ($request->input('is_master') == 'true') {
                $courses = Course::where('is_master', true)
                ->isOnline($is_online)
                ->filter($request->all())
                ->orderBy('created_at', 'ASC')
                ->paginate($request->per_page ?? 10);
            } else {
                $courses = Course::where('is_master', false)
                ->isOnline($is_online)
                ->filter($request->all())
                ->orderBy('created_at', 'ASC')
                ->paginate($request->per_page ?? 10);
            }
        }

        return CourseResource::collection($courses);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required'
        ]);
        
        $data = $request->all();
        $course = Course::create($data);

        foreach($request->input('sessions') as $session){
            $sesi = Session::create([
                'sort' => $session['sort'],
                'title' => $session['title'],
                'description' => $session['description'],
                'course_id' => $course->id,
                'live' => json_encode($session['lives'] ?? []),
                'youtube_video' => json_encode($session['videos'] ?? [])
            ]);

            foreach($session['files'] as $asset){
                File::create([
                    'session_id' => $sesi->id,
                    'asset_id' => $asset['id']
                ]);
            }
        }

        $course->users()->sync($request->users);

        return $this->successResponse();
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $course = Course::find($id);

        if($course){
            return (new CourseResource($course));
        }

        return $this->errorResponse();
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $course = Course::find($id);

        if($course){
            $course->update($request->all());

            // delete sessions
            Session::where('course_id', $course->id)->delete();

            // recreate sessions
            foreach($request->input('sessions') as $session){
                $sesi = Session::create([
                    'sort' => $session['sort'],
                    'title' => $session['title'],
                    'description' => $session['description'],
                    'course_id' => $course->id,
                    'live' => json_encode($session['lives'] ?? []),
                    'youtube_video' => json_encode($session['videos'] ?? []),
                ]);

                foreach($session['files'] as $asset){
                    File::create([
                        'session_id' => $sesi->id,
                        'asset_id' => $asset['id']
                    ]);
                }
            }

            $course->users()->sync($request->users);

            return $this->successResponse('Course berhasil diperbarui');
        }

        return $this->errorResponse('Course tidak ditemukan', 404);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $course = Course::find($id);

        if($course){
            $course->delete();

            return $this->successResponse('Course berhasil dihapus');
        }

        return $this->errorResponse();
    }
    
    public function batchDestroy(Request $request) {
        $courses = Course::whereIn('id', $request->id);
        
        if($courses->count() > 0){
            $courses->delete();
            
            return $this->successResponse('Course berhasil dihapus');
        }
        
        return $this->errorResponse();
    }
}
